#include <interface/Window.h>
#include <interface/Alert.h>
#include <be/support/String.h>
#include <be/support/Debug.h>
#include "ScanWindow.h"
#include "Sanity.h"
#include "BeSANE.h"


// --------------------------------------------------------------
ScanWindow::ScanWindow(BPoint pos)
	: BWindow(BRect(pos.x,pos.y,pos.x+360,pos.y+40), "Scanning...", B_TITLED_WINDOW_LOOK, B_MODAL_APP_WINDOW_FEEL, B_ASYNCHRONOUS_CONTROLS | B_NOT_CLOSABLE | B_NOT_RESIZABLE | B_NOT_MINIMIZABLE | B_NOT_ZOOMABLE) {
	
//	SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	
	m_statusBar = new BStatusBar(BRect(10,-4,250,16), NULL, NULL);
	m_statusBar->SetResizingMode(B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP);
	m_statusBar->SetFlags(m_statusBar->Flags() | (B_WILL_DRAW )); // | B_FULL_UPDATE_ON_RESIZE));
	AddChild(m_statusBar);
	
	m_cancelButton = new BButton(BRect(270,6,350,26),"cancel","Cancel",new BMessage(SCAN_CANCEL));
	m_cancelButton->SetResizingMode(B_FOLLOW_RIGHT | B_FOLLOW_TOP);
	m_cancelButton->ResizeToPreferred();
	SetDefaultButton(m_cancelButton);
	AddChild(m_cancelButton);
}


// --------------------------------------------------------------
ScanWindow::~ScanWindow() {
}


// --------------------------------------------------------------
bool ScanWindow::QuitRequested() {
	return true;
}


// --------------------------------------------------------------
void ScanWindow::MessageReceived(BMessage *	msg) {
	float fval;
	
	switch (msg->what){
		case SCAN_START:
			msg->FindFloat("maxtime",&fval);
			m_statusBar->SetMaxValue(fval);
			break;
		case SCAN_UPDATE:
			msg->FindFloat("addtime",&fval);
			m_statusBar->Update(fval);
			break;
		case SCAN_CANCEL:
			m_cancel = true;
			break;
		default:	
			_inherited::MessageReceived(msg);
	}
}

// --------------------------------------------------------------
void ScanWindow::Scan(SANE_Handle device) {
	m_device = device;
	m_target = LooperForThread(find_thread(NULL));
	m_image = NULL;
	m_cancel = false;
	m_scanThread = spawn_thread(doScan,"doscan",B_NORMAL_PRIORITY,(void *)this);
	resume_thread(m_scanThread);
}


// --------------------------------------------------------------
int32 ScanWindow::doScan(void *data) {
	SANE_Parameters parm;
	SANE_Status	status;
	int len;
	bool first_frame;
	ScanWindow *window = (ScanWindow *)data;
	BMessage *msg;

	// Delete any old image...
	first_frame = true;
	window->m_image = NULL;

	
	do { //  for each frame...
		// start frame reading
		if (window->m_cancel)
			goto cancel;

		status = sane_start(window->m_device);
		if ( status != SANE_STATUS_GOOD ) {
			BString text;
			text << "sane_start: ";
			text << sane_strstatus (status);
			BAlert *alert = new BAlert("Error",text.String(),"Damn!");
			alert->Go();
			goto error;
		}

		// get frame parameters
	    status = sane_get_parameters(window->m_device, &parm);
	    if (status != SANE_STATUS_GOOD) {
			BString text;
			text << "sane_get_parameters: ";
			text << sane_strstatus (status);
			BAlert *alert = new BAlert("Error",text.String(),"Damn!");
			alert->Go();
			goto error;
		}

	
		if (parm.lines >= 0) {
			if ( first_frame ) {
				msg = new BMessage(SCAN_START);
				msg->AddFloat("maxtime",(((parm.format == SANE_FRAME_RGB) || (parm.format == SANE_FRAME_GRAY)) ? 1.0 : 3.0)  * parm.bytes_per_line * parm.lines);
				window->PostMessage(msg);
				
				if ((parm.depth == 1) && (parm.format == SANE_FRAME_GRAY))
					window->m_image = new BBitmap(BRect(0, 0, parm.pixels_per_line - 1, parm.lines - 1), B_GRAY1);
				else if (parm.format == SANE_FRAME_GRAY)
					window->m_image = new BBitmap(BRect(0, 0, parm.pixels_per_line - 1, parm.lines - 1), B_RGB32);
				else
					window->m_image = new BBitmap(BRect(0, 0, parm.pixels_per_line - 1, parm.lines - 1), B_RGB32);
			}
		} else {
			BAlert *alert = new BAlert("Error:","Handscanners not supported yet!","Damn!");
			alert->Go();
			goto error;
		}
		

		uint8 * buffer, * data;
		uint8 * ptr;
		int32	x, y;
		int		channel;
		uint8	value;

		switch(parm.format) {
			case SANE_FRAME_RED:
				channel = 2;
				//window->m_statusBar->SetBarColor(red_color);
				break;
			case SANE_FRAME_GREEN:
				channel = 1;
				//window->m_statusBar->SetBarColor(green_color);
				break;
			case SANE_FRAME_BLUE:
				channel = 0;
				//window->m_statusBar->SetBarColor(blue_color);
				break;
			case SANE_FRAME_RGB:
				channel = 2;
				//window->m_statusBar->SetBarColor(default_color);
				break;
			default:
				channel = 0;
				//window->m_statusBar->SetBarColor(default_color);
				break;
		}
			
		buffer = new uint8[parm.bytes_per_line];
		x = y = 0;

		while (y < parm.lines) {	// read until end of frame or error
			if (window->m_cancel)
				goto cancel;
			status = sane_read(window->m_device, buffer, parm.bytes_per_line, &len);
			if (status == SANE_STATUS_EOF)
				break;
			if (status != SANE_STATUS_GOOD) {
				BString text;
				text << "sane_read: ";
				text << sane_strstatus (status);
				BAlert *alert = new BAlert("Error",text.String(),"Damn!");
				alert->Go();
				goto error;
			}

			msg = new BMessage(SCAN_UPDATE);
			msg->AddFloat("addtime",(float)len);
			window->PostMessage(msg);					
			
			ptr = (uint8 *)window->m_image->Bits();
			ptr += (y * window->m_image->BytesPerRow());
			if ((parm.depth == 1) && (parm.format == SANE_FRAME_GRAY))
				ptr += x/8;
			else if (parm.format == SANE_FRAME_GRAY)
				ptr += 4*x;
			else
				ptr += 4*x;

			data = buffer;

			while (len > 0) {
				switch (parm.format) {
					case SANE_FRAME_RED:
					case SANE_FRAME_GREEN:
					case SANE_FRAME_BLUE:
						if (parm.depth == 16) {
							value = (*((uint16 *) data)) >> 8;
							data += 2;
							len -= 2;
						} else {
							value = *((uint8 *) data);
							data++;
							len--;
						};

						*(ptr + channel) = value;
						*(ptr + 3) = 255;
						ptr += 4;
						x++;
						break;
						
					case SANE_FRAME_RGB:
						if (parm.depth == 16) {
							value = (*((uint16 *) data)) >> 8;
							data += 2;
							len -= 2;
						} else {
							value = *((uint8 *) data);
							data++;
							len--;
						}

						*(ptr + channel) = value;
						channel = (channel - 1);
						if (channel < 0) {
							channel = 2;
							*(ptr + 3) = 255;
							ptr += 4;
							x++;
						}
						break;
						
					case SANE_FRAME_GRAY:
						if (parm.depth == 1 ) {
							*ptr = *((uint8 *) data);
							data++;
							len--;
							ptr++;
							x+=8;
							break;
						}
						
						if (parm.depth == 16) {
							value = (*((uint16 *) data)) >> 8;
							data += 2;
							len -= 2;
						} else if ( parm.depth == 8 ) {
							value = *((uint8 *) data);
							data++;
							len--;
						}

						*ptr++ = value;	
						*ptr++ = value;	
						*ptr++ = value;	
						*ptr++ = 255;	
						x++;
						break;
				}
					
				// Next pixel;		
				if ( x >= parm.pixels_per_line ) {
					y++;
					if ( y >= parm.lines )
						break;
					x = 0;
					// skip end of row padding bytes...
					ptr = (uint8 *)window->m_image->Bits();
					ptr += (y * window->m_image->BytesPerRow());
				}
			}				
		}	// while sane_read()
				
		delete[] buffer;
			
		first_frame = false;
			
	} while ( ! parm.last_frame );

	sane_cancel(window->m_device);
	msg = new BMessage(SCAN_COMPLETE);
	msg->AddPointer("image",window->m_image);
	window->m_target->PostMessage(msg);
	window->Quit();
	return 0;

cancel:
	sane_cancel(window->m_device);
	msg = new BMessage(SCAN_CANCEL);
	window->m_target->PostMessage(msg);
	if (window->m_image) {
		delete window->m_image;
		window->m_image = NULL;
	}
	window->Quit();
	return -1;	

error:
	sane_cancel(window->m_device);
	msg = new BMessage(SCAN_ERROR);
	window->m_target->PostMessage(msg);
	if (window->m_image) {
		delete window->m_image;
		window->m_image = NULL;
	}
	window->Quit();
	return -1;	
}
