#include <be/support/String.h>
#include <be/support/ClassInfo.h>
#include <be/support/Debug.h>
#include <stdio.h>
#include <stdlib.h>

#include "Sanity.h"
#include "ControlPanel.h"
#include "BeSANE.h"

#define CONTROL_PANEL_WIDTH		290
#define CONTROL_PANEL_HEIGHT	400
#define CONTROL_ITEM_WIDTH		250
#define CONTROL_ITEM_DIVIDER	120

ControlPanel::ControlPanel(void)
	: BWindow(BRect(10,10,10+CONTROL_PANEL_WIDTH,10+CONTROL_PANEL_HEIGHT),"Control Panel", B_FLOATING_WINDOW,B_NOT_CLOSABLE | B_NOT_ZOOMABLE | B_NOT_RESIZABLE | B_ASYNCHRONOUS_CONTROLS) {
	BMenu*					menu;
	BMenuItem*				item;
	BBox*					box;
	BRect					rect;
	BTextControl*			textcontrol;
	BCheckBox*				boolcontrol;
	BMenuField*				menufield;
	SANE_Status				status;
	const SANE_Device ** 	devices_list;
	
	// No property-panel at the moment!
	m_propertyPanel = NULL;
	m_selectedDeviceHandle = NULL;
	
	status = sane_get_devices(&devices_list, SANE_TRUE);
    if (status != SANE_STATUS_GOOD) {
    	BString text;
    	text << "sane_get_devices: ";
    	text << sane_strstatus (status);
    	BAlert *alert = new BAlert("Error",text.String(),"Damn!");
    	alert->Go();
    	exit(-1);
    }
    
    int pipes[2];
    if (pipe(pipes)) {
    	BAlert *alert = new BAlert("Error",
    		"Okay, we have got a little problem. Sanity is\n"
    		"unable to create file-handles. Probably this\n"
    		"is due to a bug in BeSANE. In order to get\n"
    		"Sanity work, please remove all unneeded SANE\n"
    		"drivers from the SANE directory and then try\n"
    		"again.\n\n"
    		"If this does not help, feel free to contact\n"
    		"the author Kaya Memisoglu for help at\n"
    		"\tk.memisoglu@majix.de","Dang!");
    	alert->Go();
    	exit(-1);
    }
    close(pipes[0]);
    close(pipes[1]);
    
	// Create the tabview
	rect = Bounds();
	rect.top=34;
	m_tabView = new BTabView(rect,"propertytabs");
	AddChild(m_tabView);

	// Build the basic option tab
	rect = m_tabView->Bounds();
	m_locationPanel = new BView(rect,"Basics",B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW);
	m_locationPanel->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	m_tabView->AddTab(m_locationPanel);

	box = new BBox(BRect(4,4,CONTROL_PANEL_WIDTH-6,80));
	box->SetLabel("Boundaries");
	m_locationPanel->AddChild(box);
	textcontrol = new BTextControl(BRect(10,18,130,42),"tl-y","Upper [mm]","0",NULL);
	box->AddChild(textcontrol);	
	textcontrol = new BTextControl(BRect(150,18,260,42),"tl-x","Left [mm]","0",NULL);
	box->AddChild(textcontrol);	
	textcontrol = new BTextControl(BRect(10,44,130,68),"br-y","Lower [mm]","0",NULL);
	box->AddChild(textcontrol);	
	textcontrol = new BTextControl(BRect(150,44,260,68),"br-x","Right [mm]","0",NULL);
	box->AddChild(textcontrol);	

	box = new BBox(BRect(4,84,CONTROL_PANEL_WIDTH-6,180));
	box->SetLabel("Quality Options");
	m_locationPanel->AddChild(box);
	boolcontrol = new BCheckBox(BRect(10,18,200,42),"preview","Preview mode",NULL);
	box->AddChild(boolcontrol);
	
	
	// Build the colour option tab
	rect = m_tabView->Bounds();
	m_colourPanel = new BView(rect,"Colour",B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW);
	m_colourPanel->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	//m_tabView->AddTab(m_colourPanel);


	// And last create an empty tab for the advanced properties	
	rect = m_tabView->Bounds();
	m_propertyPanel = new BView(rect,"Advanced",B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW);
	m_propertyPanel->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	m_tabView->AddTab(m_propertyPanel);


    // Create the device menu stuff
    rect = Bounds();
    rect.bottom = 33;
    box = new BBox(rect,"background_box");
	box->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));
	AddChild(box);	

	// Build the device menu.
	// Note that this has to be the very last step, as the selected device
	// will change the constructed tabs!
	int	i;
	menu = new BMenu("Scanner Device");
	menu->SetLabelFromMarked(true);
	menu->SetRadioMode(true);
	menu->ResizeBy(100,10);
	for (i = 0; devices_list[i]; ++i) {
		BString 	label;
		BMessage * 	msg;
		
		label << devices_list[i]->name;
		label << " (" << devices_list[i]->vendor << " " << devices_list[i]->model << ")";

		msg = new BMessage(SET_DEVICE_MSG);
		msg->AddData("device", B_RAW_TYPE, devices_list[i], sizeof(devices_list[i])); 
		
		item = new BMenuItem(label.String(), msg);
		if ( i == 0 ) {
			item->SetMarked(true);
			MessageReceived(msg);
		}
			
		menu->AddItem(item);
	}

	menufield = new BMenuField(BRect(16,4,CONTROL_PANEL_WIDTH-32,26),"Device","Device",menu,true);
	menufield->SetDivider(50);
	box->AddChild(menufield);	
}


ControlPanel::~ControlPanel() {
	if (m_selectedDeviceHandle != NULL) {
		sane_close(m_selectedDeviceHandle);
		m_selectedDeviceHandle = NULL;
	}
}



bool ControlPanel::CreatePropertyPanel() {
	BView* view;
	BScrollView* scrollview;
	const SANE_Option_Descriptor *	opt;
	int i, nb_opt;
	int x, y, width, height;
	BRect rect;

	if (m_selectedDeviceHandle == NULL)
		return false;

	// Clear the old advanced tab...
	m_tabView->RemoveTab(1);
	rect = m_tabView->Bounds();
	rect.right -= B_V_SCROLL_BAR_WIDTH+1;
	rect.bottom -= 2*B_H_SCROLL_BAR_HEIGHT-4;
	m_propertyPanel = new BView(rect,"Advanced",B_FOLLOW_LEFT_RIGHT | B_FOLLOW_TOP_BOTTOM, B_WILL_DRAW);
	m_propertyPanel->SetViewColor(ui_color(B_PANEL_BACKGROUND_COLOR));

	// And reset the standard option tab
	((BTextControl *)m_locationPanel->FindView("tl-x"))->SetEnabled(false);
	((BTextControl *)m_locationPanel->FindView("tl-y"))->SetEnabled(false);
	((BTextControl *)m_locationPanel->FindView("br-x"))->SetEnabled(false);
	((BTextControl *)m_locationPanel->FindView("br-y"))->SetEnabled(false);
	((BCheckBox *)m_locationPanel->FindView("preview"))->SetEnabled(false);

	// ANd now get the number of options for this device
	sane_control_option(m_selectedDeviceHandle, 0, SANE_ACTION_GET_VALUE, &nb_opt, 0);

	// Initialize the coordinates...
	x = 10;
	y = 2;
	
	// Now we scann all options and we put them into the view.
	// We skip option 0, as this contains an option counter...
	for ( i = 1; i < nb_opt; i++ ) {
		opt = sane_get_option_descriptor(m_selectedDeviceHandle, i);
		if ( opt == NULL )
			break;
			
		if ( opt->name == NULL )
			continue;
			
		// First check for some special types!
		if (strcmp(opt->name,"tl-x") == 0) {
			EnableNumberProperty(m_locationPanel, i);
			continue;
		} else if (strcmp(opt->name,"tl-y") == 0) {
			EnableNumberProperty(m_locationPanel, i);
			continue;
		} else if (strcmp(opt->name,"br-x") == 0) {
			EnableNumberProperty(m_locationPanel, i);
			continue;
		} else if (strcmp(opt->name,"br-y") == 0) {
			EnableNumberProperty(m_locationPanel, i);
			continue;
		} else if (strcmp(opt->name,"preview") == 0) {
			EnableBoolProperty(m_locationPanel, i);
			continue;
		}

		// Now check for any other type...
		if (opt->type == SANE_TYPE_BOOL) { 
			view = CreateBoolProperty(i,x,y);
		} else if (opt->type == SANE_TYPE_BUTTON) {
			view = CreateButtonProperty(i,x,y);
		} else if (opt->constraint_type == SANE_CONSTRAINT_RANGE) {
			view = CreateRangeProperty(i,x,y);
		} else if (opt->constraint_type == SANE_CONSTRAINT_WORD_LIST) {
			view = CreateIntegerListProperty(i,x,y);
		} else if (opt->constraint_type == SANE_CONSTRAINT_STRING_LIST) {
			view = CreateStringListProperty(i,x,y);
		} else if (opt->type == SANE_TYPE_STRING) {
			view = CreateStringProperty(i,x,y);
		} else if ((opt->type == SANE_TYPE_INT) || (opt->type == SANE_TYPE_FIXED)){
			view = CreateNumberProperty(i,x,y);
		} else if (opt->type == SANE_TYPE_GROUP) {
			// view = new BStringView(BRect(x,y,x+200,y+16),opt->name,opt->title);
			view = NULL;
		} else	{
			view = new BStringView(BRect(x,y,x+200,y+16),opt->name,opt->title);
		}
		
		if (view != NULL) {
			width = int(view->Bounds().right - view->Bounds().left); 
			height = int(view->Bounds().bottom - view->Bounds().top); 
			m_propertyPanel->AddChild(view);
			y += height+2;
		}
	}

	scrollview = new BScrollView("Advanced", m_propertyPanel, B_FOLLOW_ALL_SIDES, 0, false,true, B_NO_BORDER);
	scrollview->ScrollBar(B_VERTICAL)->SetRange(0,y-rect.bottom+rect.top);
	scrollview->ScrollBar(B_VERTICAL)->SetProportion((rect.bottom-rect.top) / y);
	m_tabView->AddTab(scrollview);
	m_tabView->Invalidate();

	return true;
}



bool ControlPanel::QuitRequested(void) {
	return true;
}


void ControlPanel::MessageReceived(BMessage *	msg) {
	const SANE_Option_Descriptor *opt;
	SANE_Status	status;
	SANE_Int info;
	SANE_Bool boolValue;
	const SANE_Char* stringValue;
	SANE_Word wordValue;
	BView *view;
	BMenu *menu;
	int32 idx;

	switch (msg->what){
		case SET_DEVICE_MSG:
			ssize_t 	data_size;
			msg->FindData("device", B_RAW_TYPE, (const void **) &m_selectedDevice, &data_size);
			
			if (m_selectedDeviceHandle != NULL) {
				sane_close(m_selectedDeviceHandle);
				m_selectedDeviceHandle = NULL;
			}

			status = sane_open(m_selectedDevice->name, &m_selectedDeviceHandle);
			if (( status != SANE_STATUS_GOOD ) || (m_selectedDeviceHandle == NULL)) {
				BString text;
				text << "sane_open: ";
				text << sane_strstatus (status);
				BAlert *alert = new BAlert("Error",text.String(),"Damn!");
				alert->Go();
				m_selectedDeviceHandle = NULL;
				return;
			}
			
			CreatePropertyPanel();
			
			break;
			
		case SET_BOOL_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			boolValue = ((BCheckBox *)view)->Value() == B_CONTROL_ON ? SANE_TRUE : SANE_FALSE;
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) &boolValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;
			
		case SET_STRING_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			stringValue = ((BTextControl *)view)->Text();
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) stringValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;
			
		case SET_STRINGLIST_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			menu = ((BMenuItem *)view)->Menu();
			wordValue = menu->IndexOf((BMenuItem *)view);
			stringValue = opt->constraint.string_list[wordValue];
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) stringValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;
			
		case SET_INT_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			if (opt->type == SANE_TYPE_FIXED)
				wordValue = (int)(atof(((BTextControl *)view)->Text())*(float)(1<<16));
			else
				wordValue = atoi(((BTextControl *)view)->Text());
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) &wordValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;
			
		case SET_INTLIST_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			menu = ((BMenuItem *)view)->Menu();
			wordValue = menu->IndexOf((BMenuItem *)view);
			wordValue = opt->constraint.word_list[wordValue];
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) &wordValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;
			
		case SET_INTRANGE_PROPERTY:
			msg->FindInt32("property",(int32 *)&idx);
			msg->FindPointer("format",(void **)&opt);
			msg->FindPointer("source",(void **)&view);
			wordValue = ((BSlider *)view)->Value();
			sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_SET_VALUE,(void *) &wordValue, &info);
			if (info & SANE_INFO_RELOAD_OPTIONS)
				UpdatePropertyPanel();
			break;					
					
		default:	
			_inherited::MessageReceived(msg);
	}
}


const SANE_Device* ControlPanel::GetSelectedDevice() const {
	return m_selectedDevice;
}


SANE_Handle ControlPanel::GetSelectedDeviceHandle() const {
	return m_selectedDeviceHandle;
}


bool ControlPanel::UpdatePropertyPanel(void) {
	const SANE_Option_Descriptor *	opt;
	BView *view;
	int i, nb_opt;

	ASSERT(m_selectedDeviceHandle != NULL);
	ASSERT(m_propertyPanel != NULL);
	
	// ANd now get the number of options for this device
	sane_control_option(m_selectedDeviceHandle, 0, SANE_ACTION_GET_VALUE, &nb_opt, 0);

	// Now we scann all options and we put them into the view.
	for ( i = 1; i < nb_opt; i++ ) {
		opt = sane_get_option_descriptor(m_selectedDeviceHandle, i);
		if ( opt == NULL )
			break;
			
		if ( opt->name == NULL )
			continue;
		
		view = m_propertyPanel->FindView(opt->name);
		if (view == NULL)
			continue;
		
		if (is_kind_of(view, BControl)) {
			if (opt->cap & SANE_CAP_INACTIVE)
				((BControl *)view)->SetEnabled(false);
			else
				((BControl *)view)->SetEnabled(true);
		}
	}
	return true;	
}


BView* ControlPanel::CreateBoolProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Bool value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);

	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_BOOL_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BCheckBox *view;
	view = new BCheckBox(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+16),opt->name,opt->title,msg);
	view->SetValue(value == SANE_TRUE ? B_CONTROL_ON : B_CONTROL_OFF);
		
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);
		
	return view;
}


bool ControlPanel::EnableBoolProperty(BView* parent, int idx) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Bool value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);

	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_BOOL_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BCheckBox *view = (BCheckBox *)parent->FindView(opt->name);
	ASSERT(view != NULL);
	view->SetValue(value == SANE_TRUE ? B_CONTROL_ON : B_CONTROL_OFF);
	view->SetMessage(msg);
	view->SetEnabled(true);
		
	return true;
}


BView* ControlPanel::CreateNumberProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Word value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);
	
	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_INT_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BString text;	
	if (opt->type == SANE_TYPE_FIXED) 
		text << (float)value / (float)(1<<16);
	else
		text << value;
		
	BTextControl *view;
	view = new BTextControl(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+20),opt->name,opt->title,text.String(),msg);
	view->SetDivider(CONTROL_ITEM_DIVIDER);
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);

	return view;
}



bool ControlPanel::EnableNumberProperty(BView* parent, int idx) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Word value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);
	
	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_INT_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BTextControl *view = (BTextControl *)parent->FindView(opt->name);
	ASSERT(view != NULL);

	BString text;	
	if (opt->type == SANE_TYPE_FIXED) 
		text << (float)value / (float)(1<<16);
	else
		text << value;
		
	view->SetText(text.String());
	view->SetMessage(msg);
	view->SetEnabled(true);

	return true;
}



BView* ControlPanel::CreateRangeProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Word value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);

	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_INTRANGE_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BSlider *view;
	BString minLabel,maxLabel;
	int i;
	
	if (opt->type == SANE_TYPE_FIXED) {
		minLabel << (float)opt->constraint.range->min / (float)(1 << 16);
		maxLabel << (float)opt->constraint.range->max / (float)(1 << 16);
	} else {
		minLabel << opt->constraint.range->min;
		maxLabel << opt->constraint.range->max;
	}
	
	switch (opt->unit) {
		case SANE_UNIT_PIXEL:		/* value is number of pixels */
			minLabel << " pixels";
			maxLabel << " pixels";
			break;
   		case SANE_UNIT_BIT:			/* value is number of bits */
			minLabel << " bits";
			maxLabel << " bits";
			break;
		case SANE_UNIT_MM:			/* value is millimeters */
			minLabel << " mm";
			maxLabel << " mm";
			break;
    	case SANE_UNIT_DPI:			/* value is resolution in dots/inch */
			minLabel << " dpi";
			maxLabel << " dpi";
			break;
    	case SANE_UNIT_PERCENT:		/* value is a percentage */
			minLabel << "%";
			maxLabel << "%";
			break;
    	case SANE_UNIT_MICROSECOND:	/* value is micro seconds */
			minLabel << " msec";
			maxLabel << " msec";
			break;
		default:
			break;
	}
	
	view = new BSlider(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+40),opt->name,opt->title,msg,opt->constraint.range->min,opt->constraint.range->max);
	view->SetValue(value);
	if (opt->constraint.range->quant > 0) {
		i = (opt->constraint.range->max - opt->constraint.range->min)/opt->constraint.range->quant;
		if (i > 20) i = 20;
	} else {
		i = 20;
	}
	view->SetHashMarks(B_HASH_MARKS_BOTTOM);
	view->SetHashMarkCount(i);
	view->SetLimitLabels(minLabel.String(),maxLabel.String());
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);

	return view;
}


BView* ControlPanel::CreateIntegerListProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_Word value;
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) &value, &info);

	BMenuField *view;
	BMenu *menu;
	BMenuItem *item;
	BString label;
	int i;

	menu = new BMenu(opt->name);
	menu->SetLabelFromMarked(true);
	menu->SetRadioMode(true);
	
	for (i = 1; i<=opt->constraint.word_list[0]; ++i) {
		label = "";
		if (opt->type == SANE_TYPE_FIXED)
			label << (float)opt->constraint.word_list[i] / (float)(1<<16);
		else
			label << opt->constraint.word_list[i];

		switch (opt->unit) {
			case SANE_UNIT_PIXEL:		/* value is number of pixels */
				label << " pixels";
				break;
	   		case SANE_UNIT_BIT:			/* value is number of bits */
				label << " bits";
				break;
			case SANE_UNIT_MM:			/* value is millimeters */
				label << " mm";
				break;
	    	case SANE_UNIT_DPI:			/* value is resolution in dots/inch */
				label << " dpi";
				break;
	    	case SANE_UNIT_PERCENT:		/* value is a percentage */
				label << "%";
				break;
	    	case SANE_UNIT_MICROSECOND:	/* value is micro seconds */
				label << " msec";
				break;
		default:
			break;
		}

		BMessage *msg = new BMessage(SET_INTLIST_PROPERTY);
		msg->AddInt32("property", idx);
		msg->AddPointer("format",opt);
		
		item = new BMenuItem(label.String(), msg);
		if ( opt->constraint.word_list[i] == value ) {
			item->SetMarked(true);
		}
		menu->AddItem(item);
	}

	view = new BMenuField(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+22),opt->name,opt->title,menu);
	view->SetDivider(CONTROL_ITEM_DIVIDER);
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);

	return view;
}


BView* ControlPanel::CreateStringProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_String value = new SANE_Char[opt->size];
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) value, &info);

	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_STRING_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BTextControl *view;
	view = new BTextControl(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+20),opt->name,opt->title,value,msg);
	view->SetDivider(CONTROL_ITEM_DIVIDER);
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);

	delete[] value;
	
	return view;
}


BView* ControlPanel::CreateStringListProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// And get current value for this option...
	SANE_String value = new SANE_Char[opt->size];
	SANE_Int info;
	sane_control_option(m_selectedDeviceHandle, idx, SANE_ACTION_GET_VALUE,(void *) value, &info);

	BMenuField *view;
	BMenu *menu;
	BMenuItem *item;
	int i;

	menu = new BMenu(opt->name);
	menu->SetLabelFromMarked(true);
	menu->SetRadioMode(true);
	
	for (i = 0;; ++i) {
		if (opt->constraint.string_list[i] == NULL)
			break;

		BMessage *msg = new BMessage(SET_STRINGLIST_PROPERTY);
		msg->AddInt32("property", idx);
		msg->AddPointer("format",opt);
		
		item = new BMenuItem(opt->constraint.string_list[i], msg);
		if (strcmp(value,opt->constraint.string_list[i]) == 0 ) {
			item->SetMarked(true);
		}
		menu->AddItem(item);
	}

	view = new BMenuField(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+22),opt->name,opt->title,menu);
	((BMenuField*)view)->SetDivider(CONTROL_ITEM_DIVIDER);
	if ((opt->cap & SANE_CAP_INACTIVE) || (!(opt->cap & SANE_CAP_SOFT_SELECT)))
		view->SetEnabled(false);

	delete[] value;

	return view;
}


BView* ControlPanel::CreateButtonProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	// Create a message containing all device infos
	BMessage *msg = new BMessage(SET_BUTTON_PROPERTY);
	msg->AddInt32("property", idx);
	msg->AddPointer("format",opt);

	BView *view;
	BString label;

	label<<opt->title;
	label<<"[Button]";
	view = new BStringView(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+16),opt->name,label.String());
	return view;
}


BView* ControlPanel::CreateGroupProperty(int idx, int x, int y) {
	// Get option descriptor
	const SANE_Option_Descriptor *	opt;
	opt = sane_get_option_descriptor(m_selectedDeviceHandle, idx);

	BView *view;
	BString label;

	label<<opt->title;
	label<<"[Group]";
	view = new BStringView(BRect(x,y,x+CONTROL_ITEM_WIDTH,y+16),opt->name,label.String());
	return view;
}

