
#include <app/Application.h>
#include <InterfaceKit.h>
#include <StorageKit.h>
#include <SupportKit.h>
#include <TranslationKit.h>
#include <errno.h>

#include "Sanity.h"
#include "ScanWindow.h"
#include "MainWindow.h"
#include "BitmapView.h"

#include "BeSANE.h"

// --------------------------------------------------------------
MainWindow::MainWindow(BRect Frame)
	: BWindow(Frame, SOFTWARE_NAME, B_DOCUMENT_WINDOW, B_ASYNCHRONOUS_CONTROLS),
	m_image(NULL) {
	
	BMenuBar *	menu_bar;
	BMenu *		menu;
	BMenuItem *	item;
	BBox*		box;
	BRect		r;
	
	m_savePanel = new TranslatorSavePanel("TranslatorSavePanel",
										new BMessenger(this), NULL, 0, false,
										new BMessage(SAVE_FILE_PANEL_MSG)); 

	r = Bounds();
	
	// Okay, add our menu bar
	menu_bar = new BMenuBar(r, "menu_bar");
	AddChild(menu_bar);

	// File Menu
	menu = new BMenu(SOFTWARE_NAME);
	item = new BMenuItem("About", new BMessage(B_ABOUT_REQUESTED));
	item->SetTarget(be_app);
	menu->AddItem(item);
	menu->AddSeparatorItem();
	item = new BMenuItem("New Window", new BMessage(Sanity::NEW_WINDOW_MSG), 'N');
	item->SetTarget(be_app);
	menu->AddItem(item);
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Save As" B_UTF8_ELLIPSIS, new BMessage(SAVE_AS_MSG)));
	menu->AddSeparatorItem();
	menu->AddItem(new BMenuItem("Close", new BMessage(B_QUIT_REQUESTED), 'W'));
	item = new BMenuItem("Quit", new BMessage(B_QUIT_REQUESTED), 'Q');
	item->SetTarget(be_app);
	menu->AddItem(item);
	menu_bar->AddItem(menu);

	// Zoom Menu
	menu = new BMenu("Zoom");
	menu->SetRadioMode(true);
	item = new BMenuItem("12,5%",new BMessage(ZOOM_MSG_12));
	menu->AddItem(item);
	item = new BMenuItem("25%",new BMessage(ZOOM_MSG_25));
	menu->AddItem(item);
	item = new BMenuItem("50%",new BMessage(ZOOM_MSG_50));
	menu->AddItem(item);
	item = new BMenuItem("100%",new BMessage(ZOOM_MSG_100));
	item->SetMarked(true);
	menu->AddItem(item);
	item = new BMenuItem("200%",new BMessage(ZOOM_MSG_200));
	menu->AddItem(item);
	item = new BMenuItem("300%",new BMessage(ZOOM_MSG_300));
	menu->AddItem(item);
	item = new BMenuItem("400%",new BMessage(ZOOM_MSG_400));
	menu->AddItem(item);
	menu_bar->AddItem(menu);

	// Scan menu
	menu_bar->AddItem(new BMenuItem("Scan!",new BMessage(SCAN_MSG)));
	
	// Status bar
	r = Bounds();
	r.right -= B_V_SCROLL_BAR_WIDTH;
	r.top = r.bottom-12;
	box = new BBox(r,NULL,B_FOLLOW_LEFT_RIGHT | B_FOLLOW_BOTTOM);
	box->SetBorder(B_PLAIN_BORDER);
	AddChild(box);
	
	r = box->Bounds();
	r.top += 1;
	r.bottom += 2;
	m_statusText = new BStringView(r,"status","");
	m_statusText->SetFontSize(11.0);	
	box->AddChild(m_statusText);
	
	// The image...
	r = Bounds();
	r.top		= menu_bar->Frame().bottom+1;
	r.right 	-= B_V_SCROLL_BAR_WIDTH;
	r.bottom 	= box->Frame().top - B_H_SCROLL_BAR_HEIGHT - 2;
	
	m_bitmapView 		= new BitmapView(r, NULL);
	BScrollView *	sv 	= new BScrollView("bitmap_scroll", m_bitmapView,
										B_FOLLOW_ALL_SIDES, B_WILL_DRAW, true, true);
	AddChild(sv);
	
	UpdateStatus();
}


// --------------------------------------------------------------
MainWindow::~MainWindow() {
	delete m_savePanel;
	delete m_image;
}


// --------------------------------------------------------------
bool MainWindow::QuitRequested() {
	be_app->PostMessage(Sanity::WINDOW_CLOSED_MSG);
	return true;
}


// --------------------------------------------------------------
void MainWindow::UpdateStatus() {
	if (m_image == NULL) {
		m_statusText->SetText("No image");
	} else {
		BString text;
		text 
			<< "Image size is "
			<< int(m_image->Bounds().right - m_image->Bounds().left + 1)
			<< "x"
			<< int(m_image->Bounds().bottom - m_image->Bounds().top + 1);
		switch(m_image->ColorSpace()) {
			case B_GRAY1:
				text << " pixels in 1bit black and white";
				break;
			case B_GRAY8:
				text << " pixels in 8bit grayscale";
				break;
			case B_RGB32:
				text << " pixels in 32bit truecolour";
				break;
			default:
				text << " pixels";
		}
		
		text << "    Total " << m_image->BitsLength()/1024 << "kBytes";
			
		m_statusText->SetText(text.String());
	}
}


// --------------------------------------------------------------
void MainWindow::MessageReceived(BMessage *	msg) {
	BBitmap* image = NULL;
	
	switch (msg->what){
		case SAVE_AS_MSG:
			doSaveAs();
			break;
						
		case SAVE_FILE_PANEL_MSG:
			doFilePanel(msg);
			break;
			
		case SCAN_MSG:
			((Sanity *)be_app)->Scan();
			break;
			
		case ScanWindow::SCAN_COMPLETE:
			msg->FindPointer("image",(void **)&image);
			ASSERT(image != NULL);
			m_bitmapView->SetBitmap(image);
			if ( m_image )
				delete m_image;
			m_image = image;
			UpdateStatus();
			break;
			
		case ZOOM_MSG_12:
			m_bitmapView->SetZoom(0.125);
			break;

		case ZOOM_MSG_25:
			m_bitmapView->SetZoom(0.25);
			break;

		case ZOOM_MSG_50:
			m_bitmapView->SetZoom(0.5);
			break;

		case ZOOM_MSG_100:
			m_bitmapView->SetZoom(1.0);
			break;

		case ZOOM_MSG_200:
			m_bitmapView->SetZoom(2.0);
			break;

		case ZOOM_MSG_300:
			m_bitmapView->SetZoom(3.0);
			break;

		case ZOOM_MSG_400:
			m_bitmapView->SetZoom(4.0);
			break;
					
		default:	
			_inherited::MessageReceived(msg);
	}
}



// --------------------------------------------------------------
void MainWindow::doScan(void) {
}



// Save As
void MainWindow::doSaveAs(void) {
	if ( ! m_image ) {
		BAlert *alert = new BAlert(NULL, "No image to save.", "Doh!",
							NULL, NULL, B_WIDTH_AS_USUAL, B_STOP_ALERT);
		alert->Go();
		return;
	}
/*				
	if ( ! m_savePanel )
		m_savePanel = new TranslatorSavePanel("TranslatorSavePanel",
												new BMessenger(this), NULL, 0, false,
												new BMessage(SAVE_FILE_PANEL_MSG)); 
*/
	m_savePanel->Window()->SetWorkspaces(B_CURRENT_WORKSPACE);
	m_savePanel->Show();
}




// Other save stuff???
void MainWindow::doFilePanel(BMessage* msg) {
	// Recover the necessary data from the message
	translator_id * id;
	uint32 			format;
	ssize_t 		length = sizeof(translator_id);

	if ( msg->FindData("translator_id", B_RAW_TYPE, (const void **)&id, &length) != B_OK)
		return;
	if ( msg->FindInt32("translator_format", (int32 *)&format) != B_OK)
		return;

	entry_ref dir;
	if ( msg->FindRef("directory", &dir) != B_OK)
		return;

	BDirectory bdir(&dir);
	const char *name;

	if ( msg->FindString("name", &name) != B_OK)
		return;
	if ( name == NULL )
		return;

	// Clobber any existing file or create a new one if it didn't exist
	BFile file(&bdir, name, B_WRITE_ONLY | B_CREATE_FILE | B_ERASE_FILE);
	if (file.InitCheck() != B_OK) {
		BAlert *alert = new BAlert(NULL, "Could not create file.", "OK");
		alert->Go();
		return;
	}

	BTranslatorRoster * roster = BTranslatorRoster::Default();
	BBitmapStream stream(m_image);

	// If the id is no longer valid or the translator fails for any other
	// reason, catch it here
	if ( roster->Translate(*id, &stream, NULL, &file, format) != B_OK) {
		BAlert * alert = new BAlert(NULL, "Could not save the image.", "OK");
		alert->Go();
	}

	// Reclaim the ownership of the bitmap, otherwise it would be deleted
	// when stream passes out of scope
	stream.DetachBitmap(&m_image);
}
